// Datos de ejemplo de papeletas de los últimos 7 días
const papeletasData = {
    // Generar datos de ejemplo para los últimos 7 días
    ultimos7Dias: (() => {
        const datos = [];
        const tipos = ['Exceso de velocidad', 'Estacionamiento', 'Luz roja', 'Documentos', 'Otros'];
        const hoy = new Date();
        
        // Definir rangos para mantener un promedio cercano a 4500
        const minPapeletas = 4200;  // Mínimo de papeletas por día
        const maxPapeletas = 4800;  // Máximo de papeletas por día
        
        for (let i = 6; i >= 0; i--) {
            const fecha = new Date(hoy);
            fecha.setDate(hoy.getDate() - i);
            
            // Generar cantidad de papeletas dentro del rango objetivo
            const cantidadPapeletas = Math.floor(
                Math.random() * (maxPapeletas - minPapeletas + 1) + minPapeletas
            );
            
            // Distribuir las papeletas entre los diferentes tipos
            const distribucionTipos = tipos.map(() => Math.random());
            const sumaDistribucion = distribucionTipos.reduce((a, b) => a + b, 0);
            const papeletasPorTipo = distribucionTipos.map(valor => 
                Math.floor((valor / sumaDistribucion) * cantidadPapeletas)
            );
            
            datos.push({
                fecha: fecha.toISOString().split('T')[0],
                total: cantidadPapeletas,
                detalles: tipos.map((tipo, index) => ({
                    tipo: tipo,
                    cantidad: papeletasPorTipo[index]
                }))
            });
        }
        return datos;
    })()
};

// Función para inicializar los gráficos
function initializeCharts() {
    // Gráfico de papeletas de los últimos 7 días
    const ctx = document.getElementById('papeletasRecientes');
    if (!ctx) return;

    const datos = papeletasData.ultimos7Dias;
    const fechas = datos.map(d => {
        const fecha = new Date(d.fecha);
        return fecha.toLocaleDateString('es-ES', { weekday: 'short', day: 'numeric' });
    });
    
    const totales = datos.map(d => d.total);
    
    // Calcular el total acumulado y el promedio
    const totalAcumulado = totales.reduce((sum, current) => sum + current, 0);
    const promedioDiario = (totalAcumulado / 7).toFixed(1);

    // Actualizar el resumen de estadísticas
    document.getElementById('totalPapeletas').textContent = totalAcumulado.toLocaleString('es-PE');
    document.getElementById('promedioDiario').textContent = Number(promedioDiario).toLocaleString('es-PE');

    // Crear el gráfico
    new Chart(ctx, {
        type: 'bar',
        data: {
            labels: fechas,
            datasets: [
                {
                    label: 'Papeletas por día',
                    data: totales,
                    backgroundColor: 'rgba(33, 150, 243, 0.7)',
                    borderColor: 'rgb(33, 150, 243)',
                    borderWidth: 1,
                    borderRadius: 4,
                    maxBarThickness: 50
                },
                {
                    label: 'Tendencia',
                    type: 'line',
                    data: totales,
                    borderColor: 'rgba(255, 87, 34, 0.8)',
                    borderWidth: 2,
                    fill: false,
                    tension: 0.4,
                    pointRadius: 4,
                    pointBackgroundColor: 'rgba(255, 87, 34, 1)'
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'top',
                    labels: {
                        usePointStyle: true,
                        padding: 15
                    }
                },
                title: {
                    display: true,
                    text: 'Papeletas Emitidas - Últimos 7 días',
                    font: {
                        size: 16,
                        weight: 'bold'
                    },
                    padding: {
                        top: 10,
                        bottom: 20
                    }
                },
                tooltip: {
                    mode: 'index',
                    intersect: false,
                    callbacks: {
                        label: function(context) {
                            let label = context.dataset.label || '';
                            if (label) {
                                label += ': ';
                            }
                            label += Number(context.parsed.y).toLocaleString('es-PE') + ' papeletas';
                            return label;
                        }
                    }
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    grid: {
                        drawBorder: false,
                        color: 'rgba(0, 0, 0, 0.1)'
                    },
                    ticks: {
                        callback: function(value) {
                            return value.toLocaleString('es-PE');
                        }
                    },
                    title: {
                        display: true,
                        text: 'Cantidad de Papeletas'
                    }
                },
                x: {
                    grid: {
                        display: false
                    },
                    title: {
                        display: true,
                        text: 'Día'
                    }
                }
            },
            interaction: {
                intersect: false,
                mode: 'index'
            }
        }
    });
}

// Inicializar cuando el DOM esté listo
document.addEventListener('DOMContentLoaded', initializeCharts); 