// Función para generar fecha aleatoria entre 2017 y 2025
function generarFechaAleatoria() {
    const inicio = new Date('2017-01-01').getTime();
    const fin = new Date('2025-12-31').getTime();
    const fechaAleatoria = new Date(inicio + Math.random() * (fin - inicio));
    return fechaAleatoria.toLocaleDateString('es-PE', {
        year: 'numeric',
        month: '2-digit',
        day: '2-digit'
    });
}

// Datos de ejemplo de comunicaciones (100 registros)
const comunicaciones = [
    { fecha: '15/03/2023', placa: 'ABC-123', nombre: 'Juan Pérez', telefono: '+51 987 654 321', medio: 'WhatsApp', estado: 'recibido' },
    { fecha: '22/07/2024', placa: 'XYZ-789', nombre: 'María García', telefono: '+51 987 654 322', medio: 'Llamada', estado: 'rechazado' },
    { fecha: '05/11/2022', placa: 'DEF-456', nombre: 'Carlos López', telefono: '+51 987 654 323', medio: 'SMS', estado: 'invalido' },
    { fecha: '18/01/2023', placa: 'GHI-789', nombre: 'Ana Martínez', telefono: '+51 987 654 324', medio: 'WhatsApp', estado: 'recibido' },
    { fecha: '30/09/2024', placa: 'JKL-012', nombre: 'Pedro Sánchez', telefono: '+51 987 654 325', medio: 'Llamada', estado: 'rechazado' },
    { fecha: '12/04/2023', placa: 'MNO-345', nombre: 'Laura Torres', telefono: '+51 987 654 326', medio: 'SMS', estado: 'invalido' },
    { fecha: '25/08/2024', placa: 'PQR-678', nombre: 'Diego Flores', telefono: '+51 987 654 327', medio: 'WhatsApp', estado: 'recibido' },
    { fecha: '03/12/2022', placa: 'STU-901', nombre: 'Carmen Ruiz', telefono: '+51 987 654 328', medio: 'Llamada', estado: 'rechazado' },
    { fecha: '14/06/2023', placa: 'VWX-234', nombre: 'José Ramírez', telefono: '+51 987 654 329', medio: 'SMS', estado: 'invalido' },
    { fecha: '28/02/2024', placa: 'YZA-567', nombre: 'Isabel Castro', telefono: '+51 987 654 330', medio: 'WhatsApp', estado: 'recibido' },
    // ... Continuar con más registros hasta completar 100
];

// Generar datos adicionales para completar 100 registros
const nombres = ['Roberto', 'Patricia', 'Miguel', 'Sofía', 'Fernando', 'Lucía', 'Andrés', 'Valentina', 'Gabriel', 'Carolina'];
const apellidos = ['González', 'Rodríguez', 'Hernández', 'López', 'Martínez', 'Díaz', 'Torres', 'Vargas', 'Silva', 'Rojas'];
const medios = ['WhatsApp', 'Llamada', 'SMS'];
const estados = ['recibido', 'rechazado', 'invalido'];

function generarPlaca() {
    const letras = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    const numeros = '0123456789';
    return `${letras[Math.floor(Math.random() * letras.length)]}${letras[Math.floor(Math.random() * letras.length)]}${letras[Math.floor(Math.random() * letras.length)]}-${numeros[Math.floor(Math.random() * numeros.length)]}${numeros[Math.floor(Math.random() * numeros.length)]}${numeros[Math.floor(Math.random() * numeros.length)]}`;
}

function generarTelefono() {
    return '+51 9' + Array.from({length: 8}, () => Math.floor(Math.random() * 10)).join('');
}

// Completar los registros hasta 100
while (comunicaciones.length < 100) {
    comunicaciones.push({
        fecha: generarFechaAleatoria(),
        placa: generarPlaca(),
        nombre: `${nombres[Math.floor(Math.random() * nombres.length)]} ${apellidos[Math.floor(Math.random() * apellidos.length)]}`,
        telefono: generarTelefono(),
        medio: medios[Math.floor(Math.random() * medios.length)],
        estado: estados[Math.floor(Math.random() * estados.length)]
    });
}

class ComunicacionesUI {
    constructor() {
        this.currentPage = 1;
        this.itemsPerPage = 10;
        this.filteredData = [...comunicaciones];
        
        // Inicializar después de que el DOM esté listo
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => this.init());
        } else {
            this.init();
        }
    }

    init() {
        // Inicializar referencias a elementos del DOM
        this.initializeDOMReferences();
        
        // Solo continuar si tenemos los elementos necesarios
        if (this.checkRequiredElements()) {
            // Inicializar componentes MDC
            this.initializeMDC();
            
            // Configurar event listeners
            this.setupEventListeners();
            
            // Renderizar datos iniciales
            this.renderTable();

            // Inicializar indicador de scroll si existe el contenedor
            if (this.tableContainer) {
                this.initScrollIndicator();
            }

            // Configurar el sidebar
            this.setupSidebar();

            // Configurar el scroll del top app bar
            this.setupTopAppBarScroll();
        } else {
            console.error('No se encontraron algunos elementos requeridos del DOM');
        }
    }

    initializeDOMReferences() {
        // Referencias a elementos base
        this.topAppBar = document.querySelector('.mdc-top-app-bar');
        this.mainContent = document.querySelector('.main-content');

        // Si no existe el contenido principal, crearlo
        if (!this.mainContent) {
            this.mainContent = document.createElement('main');
            this.mainContent.className = 'main-content';
            document.body.appendChild(this.mainContent);
        }

        // Crear sección de filtros si no existe
        let filtersSection = this.mainContent.querySelector('.filters-section');
        if (!filtersSection) {
            filtersSection = document.createElement('div');
            filtersSection.className = 'filters-section';
            filtersSection.innerHTML = `
                <div class="search-field mdc-text-field mdc-text-field--outlined">
                    <input type="text" id="searchComunicacion" class="mdc-text-field__input">
                    <div class="mdc-notched-outline">
                        <div class="mdc-notched-outline__leading"></div>
                        <div class="mdc-notched-outline__notch">
                            <label class="mdc-floating-label">Buscar...</label>
                        </div>
                        <div class="mdc-notched-outline__trailing"></div>
                    </div>
                </div>
                <div class="mdc-select mdc-select--outlined">
                    <div class="mdc-select__anchor">
                        <span class="mdc-select__selected-text"></span>
                        <div class="mdc-notched-outline">
                            <div class="mdc-notched-outline__leading"></div>
                            <div class="mdc-notched-outline__notch">
                                <label class="mdc-floating-label">Medio</label>
                            </div>
                            <div class="mdc-notched-outline__trailing"></div>
                        </div>
                    </div>
                    <div class="mdc-select__menu mdc-menu mdc-menu-surface">
                        <ul class="mdc-list">
                            <li class="mdc-list-item mdc-list-item--selected" data-value="todos" aria-selected="true">
                                <span class="mdc-list-item__ripple"></span>
                                <span class="mdc-list-item__text">Todos</span>
                            </li>
                            <li class="mdc-list-item" data-value="WhatsApp">
                                <span class="mdc-list-item__ripple"></span>
                                <span class="mdc-list-item__text">WhatsApp</span>
                            </li>
                            <li class="mdc-list-item" data-value="Llamada">
                                <span class="mdc-list-item__ripple"></span>
                                <span class="mdc-list-item__text">Llamada</span>
                            </li>
                            <li class="mdc-list-item" data-value="SMS">
                                <span class="mdc-list-item__ripple"></span>
                                <span class="mdc-list-item__text">SMS</span>
                            </li>
                        </ul>
                    </div>
                </div>
                <div class="mdc-select mdc-select--outlined">
                    <div class="mdc-select__anchor">
                        <span class="mdc-select__selected-text"></span>
                        <div class="mdc-notched-outline">
                            <div class="mdc-notched-outline__leading"></div>
                            <div class="mdc-notched-outline__notch">
                                <label class="mdc-floating-label">Estado</label>
                            </div>
                            <div class="mdc-notched-outline__trailing"></div>
                        </div>
                    </div>
                    <div class="mdc-select__menu mdc-menu mdc-menu-surface">
                        <ul class="mdc-list">
                            <li class="mdc-list-item mdc-list-item--selected" data-value="todos" aria-selected="true">
                                <span class="mdc-list-item__ripple"></span>
                                <span class="mdc-list-item__text">Todos</span>
                            </li>
                            <li class="mdc-list-item" data-value="recibido">
                                <span class="mdc-list-item__ripple"></span>
                                <span class="mdc-list-item__text">Recibido</span>
                            </li>
                            <li class="mdc-list-item" data-value="rechazado">
                                <span class="mdc-list-item__ripple"></span>
                                <span class="mdc-list-item__text">Rechazado</span>
                            </li>
                            <li class="mdc-list-item" data-value="invalido">
                                <span class="mdc-list-item__ripple"></span>
                                <span class="mdc-list-item__text">Inválido</span>
                            </li>
                        </ul>
                    </div>
                </div>
            `;
            this.mainContent.insertBefore(filtersSection, this.mainContent.firstChild);
        }

        // Crear o encontrar el contenedor de la tabla
        this.tableContainer = this.mainContent.querySelector('.table-container');
        if (!this.tableContainer) {
            this.tableContainer = document.createElement('div');
            this.tableContainer.className = 'table-container mdc-data-table';
            
            // Crear tabla
            const tableHTML = `
                <table class="mdc-data-table__table">
                    <thead>
                        <tr class="mdc-data-table__header-row">
                            <th class="mdc-data-table__header-cell" role="columnheader" scope="col">Fecha</th>
                            <th class="mdc-data-table__header-cell" role="columnheader" scope="col">Placa</th>
                            <th class="mdc-data-table__header-cell" role="columnheader" scope="col">Nombre</th>
                            <th class="mdc-data-table__header-cell" role="columnheader" scope="col">Teléfono</th>
                            <th class="mdc-data-table__header-cell" role="columnheader" scope="col">Medio</th>
                            <th class="mdc-data-table__header-cell" role="columnheader" scope="col">Estado</th>
                            <th class="mdc-data-table__header-cell" role="columnheader" scope="col">Acciones</th>
                        </tr>
                    </thead>
                    <tbody id="comunicacionesTableBody" class="mdc-data-table__content"></tbody>
                </table>
            `;
            
            this.tableContainer.innerHTML = tableHTML;
            this.mainContent.appendChild(this.tableContainer);
        }

        // Crear paginación si no existe
        let pagination = this.mainContent.querySelector('.pagination');
        if (!pagination) {
            pagination = document.createElement('div');
            pagination.className = 'pagination';
            pagination.innerHTML = `
                <button class="mdc-icon-button material-icons" aria-label="Primera página">
                    <div class="mdc-icon-button__ripple"></div>
                    first_page
                </button>
                <button class="mdc-icon-button material-icons" aria-label="Página anterior">
                    <div class="mdc-icon-button__ripple"></div>
                    chevron_left
                </button>
                <div class="pagination-info">Página 1 de 1</div>
                <button class="mdc-icon-button material-icons" aria-label="Página siguiente">
                    <div class="mdc-icon-button__ripple"></div>
                    chevron_right
                </button>
                <button class="mdc-icon-button material-icons" aria-label="Última página">
                    <div class="mdc-icon-button__ripple"></div>
                    last_page
                </button>
            `;
            this.mainContent.appendChild(pagination);
        }

        // Actualizar referencias
        this.searchField = document.getElementById('searchComunicacion');
        this.medioSelect = document.querySelector('.mdc-select:nth-of-type(1)');
        this.estadoSelect = document.querySelector('.mdc-select:nth-of-type(2)');
        this.drawer = document.querySelector('.mdc-drawer');
        this.tableBody = document.getElementById('comunicacionesTableBody');
    }

    checkRequiredElements() {
        // Solo verificar elementos esenciales
        return this.topAppBar && this.mainContent;
    }

    initializeMDC() {
        try {
            // Inicializar campo de búsqueda
            const searchFieldElement = document.querySelector('.search-field');
            if (searchFieldElement) {
                this.searchFieldMDC = new mdc.textField.MDCTextField(searchFieldElement);
            }

            // Inicializar selectores
            const selects = document.querySelectorAll('.mdc-select');
            selects.forEach((select, index) => {
                const mdcSelect = new mdc.select.MDCSelect(select);
                if (index === 0) this.medioSelectMDC = mdcSelect;
                if (index === 1) this.estadoSelectMDC = mdcSelect;
            });

            // Inicializar botones con ripple
            document.querySelectorAll('.mdc-icon-button').forEach(button => {
                new mdc.ripple.MDCRipple(button).unbounded = true;
            });

            // Drawer y Top App Bar
            if (this.drawer) {
                this.drawerComponent = new mdc.drawer.MDCDrawer(this.drawer);
            }

            if (this.topAppBar) {
                this.topAppBarComponent = new mdc.topAppBar.MDCTopAppBar(this.topAppBar);
                if (this.mainContent) {
                    this.topAppBarComponent.setScrollTarget(this.mainContent);
                    this.topAppBarComponent.fixedAdjustElement = this.mainContent;
                }
            }
        } catch (error) {
            console.error('Error al inicializar componentes MDC:', error);
        }
    }

    setupEventListeners() {
        // Búsqueda
        this.searchField.addEventListener('input', () => {
            this.currentPage = 1;
            this.filterData();
        });

        // Filtros de medio y estado
        this.medioSelectMDC.listen('MDCSelect:change', () => {
            this.currentPage = 1;
            this.filterData();
        });

        this.estadoSelectMDC.listen('MDCSelect:change', () => {
            this.currentPage = 1;
            this.filterData();
        });

        // Paginación
        document.querySelectorAll('.pagination .mdc-icon-button').forEach(button => {
            button.addEventListener('click', (e) => this.handlePagination(e));
        });
    }

    filterData() {
        const searchTerm = this.searchField.value.toLowerCase();
        const selectedMedio = this.medioSelectMDC.value;
        const selectedEstado = this.estadoSelectMDC.value;

        this.filteredData = comunicaciones.filter(item => {
            const matchesSearch = !searchTerm || 
                item.placa.toLowerCase().includes(searchTerm) ||
                item.nombre.toLowerCase().includes(searchTerm) ||
                item.telefono.includes(searchTerm) ||
                item.fecha.includes(searchTerm);

            const matchesMedio = selectedMedio === 'todos' || item.medio === selectedMedio;
            const matchesEstado = selectedEstado === 'todos' || item.estado === selectedEstado;

            return matchesSearch && matchesMedio && matchesEstado;
        });

        this.renderTable();
    }

    renderTable() {
        if (!this.tableBody) return;

        const start = (this.currentPage - 1) * this.itemsPerPage;
        const end = start + this.itemsPerPage;
        const paginatedData = this.filteredData.slice(start, end);

        this.tableBody.innerHTML = paginatedData.map(item => `
            <tr class="mdc-data-table__row">
                <td class="mdc-data-table__cell">${item.fecha}</td>
                <td class="mdc-data-table__cell">${item.placa}</td>
                <td class="mdc-data-table__cell">${item.nombre}</td>
                <td class="mdc-data-table__cell">${item.telefono}</td>
                <td class="mdc-data-table__cell">
                    <span class="status-chip ${item.medio.toLowerCase()}">
                        ${item.medio}
                    </span>
                </td>
                <td class="mdc-data-table__cell">
                    <span class="status-chip ${item.estado.toLowerCase()}">
                        ${item.estado}
                    </span>
                </td>
                <td class="mdc-data-table__cell">
                    <button class="mdc-icon-button material-icons" title="Ver detalles">visibility</button>
                    <button class="mdc-icon-button material-icons" title="Editar">edit</button>
                    <button class="mdc-icon-button material-icons" title="Eliminar">delete</button>
                </td>
            </tr>
        `).join('');

        // Actualizar información de paginación
        const totalPages = Math.ceil(this.filteredData.length / this.itemsPerPage);
        const paginationInfo = document.querySelector('.pagination-info');
        if (paginationInfo) {
            paginationInfo.textContent = `Página ${this.currentPage} de ${totalPages}`;
        }

        // Actualizar estado de botones de paginación
        const paginationButtons = document.querySelectorAll('.pagination .mdc-icon-button');
        if (paginationButtons.length === 4) {
            const [firstPage, prevPage, nextPage, lastPage] = paginationButtons;
            firstPage.disabled = this.currentPage === 1;
            prevPage.disabled = this.currentPage === 1;
            nextPage.disabled = this.currentPage === totalPages;
            lastPage.disabled = this.currentPage === totalPages;
        }

        // Reinicializar los botones de acción
        document.querySelectorAll('.mdc-icon-button').forEach(button => {
            new mdc.ripple.MDCRipple(button).unbounded = true;
        });
    }

    handlePagination(event) {
        const totalPages = Math.ceil(this.filteredData.length / this.itemsPerPage);
        const button = event.target;

        if (button.textContent === 'first_page') {
            this.currentPage = 1;
        } else if (button.textContent === 'chevron_left') {
            this.currentPage = Math.max(1, this.currentPage - 1);
        } else if (button.textContent === 'chevron_right') {
            this.currentPage = Math.min(totalPages, this.currentPage + 1);
        } else if (button.textContent === 'last_page') {
            this.currentPage = totalPages;
        }

        this.renderTable();
    }

    initScrollIndicator() {
        const checkScroll = () => {
            const hasHorizontalScroll = this.tableContainer.scrollWidth > this.tableContainer.clientWidth;
            this.tableContainer.classList.toggle('has-scroll', hasHorizontalScroll);
        };

        // Verificar scroll inicial
        checkScroll();

        // Verificar cuando cambie el tamaño de la ventana
        window.addEventListener('resize', checkScroll);

        // Verificar durante el scroll
        this.tableContainer.addEventListener('scroll', () => {
            const isAtEnd = this.tableContainer.scrollLeft + this.tableContainer.clientWidth >= this.tableContainer.scrollWidth - 10;
            this.tableContainer.classList.toggle('at-end', isAtEnd);
        });
    }

    setupSidebar() {
        // Manejar el botón de menú
        this.topAppBarComponent.listen('MDCTopAppBar:nav', () => {
            this.drawerComponent.open = !this.drawerComponent.open;
        });

        // Cerrar el drawer al hacer clic en el scrim (fondo oscuro)
        const scrim = document.querySelector('.mdc-drawer-scrim');
        if (scrim) {
            scrim.addEventListener('click', () => {
                this.drawerComponent.open = false;
            });
        }

        // Cerrar el drawer al seleccionar un elemento del menú en móvil
        const drawerItems = document.querySelectorAll('.mdc-drawer .mdc-list-item');
        drawerItems.forEach(item => {
            item.addEventListener('click', () => {
                if (window.innerWidth < 1024) {
                    this.drawerComponent.open = false;
                }
            });
        });

        // Manejar cambios de tamaño de ventana
        window.addEventListener('resize', () => {
            if (window.innerWidth >= 1024) {
                this.drawerComponent.open = true;
            } else {
                this.drawerComponent.open = false;
            }
        });
    }

    setupTopAppBarScroll() {
        // Solo agregar sombra en scroll, no mover la barra
        const handleScroll = () => {
            if (this.mainContent.scrollTop > 0) {
                this.topAppBar.classList.add('mdc-top-app-bar--scrolled');
            } else {
                this.topAppBar.classList.remove('mdc-top-app-bar--scrolled');
            }
        };

        // Escuchar el scroll en el contenido principal
        if (this.mainContent) {
            this.mainContent.addEventListener('scroll', handleScroll);
        }

        // Asegurar que la barra superior permanezca fija
        if (this.topAppBarComponent) {
            this.topAppBarComponent.listen('MDCTopAppBar:nav', () => {
                this.drawerComponent.open = !this.drawerComponent.open;
            });
        }
    }

    renderCobrador(cobrador) {
        return `
            <div class="cobrador-card">
                <div class="cobrador-header">
                    <div class="cobrador-avatar">
                        ${cobrador.nombre.charAt(0)}
                    </div>
                    <div class="cobrador-info">
                        <h3 class="cobrador-name">${cobrador.nombre}</h3>
                        <p class="cobrador-zone">${cobrador.zona}</p>
                    </div>
                    <button class="mdc-icon-button material-icons">more_vert</button>
                </div>
                <div class="main-stats">
                    <div class="total-amount">
                        <div class="amount">S/ ${cobrador.total.toFixed(2)}</div>
                        <div class="label">Total</div>
                    </div>
                    <div class="secondary-stats">
                        <div class="stat-item">
                            <div class="value">${cobrador.cobranzas}</div>
                            <div class="label">Cobranzas</div>
                        </div>
                        <div class="stat-item">
                            <div class="value">${cobrador.efectividad}%</div>
                            <div class="label">Efectividad</div>
                        </div>
                    </div>
                </div>
                <div class="success-rate">
                    <div class="success-rate-fill" style="width: ${cobrador.efectividad}%"></div>
                </div>
            </div>
        `;
    }
}

class EstadisticasComunicaciones {
    constructor(comunicaciones) {
        this.comunicaciones = comunicaciones;
        // Esperar a que el DOM esté completamente cargado
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => this.init());
        } else {
            this.init();
        }
    }

    init() {
        // Asegurarse de que los contenedores de los gráficos existan
        if (this.checkChartContainers()) {
            this.initializeCharts();
            this.updateStats();
        }
    }

    checkChartContainers() {
        const containers = [
            'graficoSemanal',
            'graficoMensual',
            'comunicacionesPorMedio',
            'comunicacionesPorEstado'
        ];
        return containers.every(id => document.getElementById(id) !== null);
    }

    initializeCharts() {
        // Configuración común para todos los gráficos
        Chart.defaults.responsive = true;
        Chart.defaults.maintainAspectRatio = false;

        // Gráfico semanal
        this.graficoSemanal = new Chart(document.getElementById('graficoSemanal'), {
            type: 'line',
            data: {
                labels: ['Hace 3 semanas', 'Hace 2 semanas', 'Última semana'],
                datasets: [
                    { 
                        label: 'Recibidas',
                        data: [],
                        borderColor: '#4CAF50',
                        tension: 0.1,
                        fill: false
                    },
                    { 
                        label: 'Rechazadas',
                        data: [],
                        borderColor: '#F44336',
                        tension: 0.1,
                        fill: false
                    },
                    { 
                        label: 'Inválidas',
                        data: [],
                        borderColor: '#9E9E9E',
                        tension: 0.1,
                        fill: false
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: 'Tendencia Semanal',
                        padding: 20
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });

        // Gráfico mensual
        this.graficoMensual = new Chart(document.getElementById('graficoMensual'), {
            type: 'line',
            data: {
                labels: ['Hace 3 meses', 'Hace 2 meses', 'Último mes'],
                datasets: [
                    { 
                        label: 'Recibidas',
                        data: [],
                        borderColor: '#4CAF50',
                        tension: 0.1,
                        fill: false
                    },
                    { 
                        label: 'Rechazadas',
                        data: [],
                        borderColor: '#F44336',
                        tension: 0.1,
                        fill: false
                    },
                    { 
                        label: 'Inválidas',
                        data: [],
                        borderColor: '#9E9E9E',
                        tension: 0.1,
                        fill: false
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: 'Tendencia Mensual',
                        padding: 20
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });

        // Gráfico por medio de comunicación
        this.graficoPorMedio = new Chart(document.getElementById('comunicacionesPorMedio'), {
            type: 'doughnut',
            data: {
                labels: ['WhatsApp', 'Llamada', 'SMS'],
                datasets: [{
                    data: [],
                    backgroundColor: ['#25D366', '#2196F3', '#FF9800']
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: 'Comunicaciones por Medio',
                        padding: 20
                    }
                }
            }
        });

        // Gráfico por estado
        this.graficoPorEstado = new Chart(document.getElementById('comunicacionesPorEstado'), {
            type: 'doughnut',
            data: {
                labels: ['Recibido', 'Rechazado', 'Inválido'],
                datasets: [{
                    data: [],
                    backgroundColor: ['#4CAF50', '#F44336', '#9E9E9E']
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: 'Comunicaciones por Estado',
                        padding: 20
                    }
                }
            }
        });
    }

    updateStats() {
        const hoy = new Date();
        const unDia = 24 * 60 * 60 * 1000;
        const unaSemana = 7 * unDia;
        const unMes = 30 * unDia;

        // Estadísticas generales
        const total = this.comunicaciones.length;
        const comunicacionesHoy = this.comunicaciones.filter(c => 
            new Date(this.parseFecha(c.fecha)).toDateString() === hoy.toDateString()
        ).length;

        const conteoMedios = this.contarPorPropiedad('medio');
        const metodoPrincipal = Object.entries(conteoMedios)
            .sort((a, b) => b[1] - a[1])[0][0];

        // Contar comunicaciones por estado
        const recibidas = this.comunicaciones.filter(c => c.estado === 'recibido').length;
        const rechazadas = this.comunicaciones.filter(c => c.estado === 'rechazado').length;
        const invalidas = this.comunicaciones.filter(c => c.estado === 'invalido').length;
        
        const tasaExito = ((recibidas / total) * 100).toFixed(1);

        // Actualizar tarjetas de estadísticas generales
        document.getElementById('totalComunicaciones').textContent = total;
        document.getElementById('comunicacionesHoy').textContent = comunicacionesHoy;
        document.getElementById('metodoPrincipal').textContent = metodoPrincipal;
        document.getElementById('tasaExito').textContent = tasaExito + '%';

        // Actualizar tarjetas de estado
        document.getElementById('totalRecibidas').textContent = recibidas;
        document.getElementById('totalRechazadas').textContent = rechazadas;
        document.getElementById('totalInvalidas').textContent = invalidas;

        // Actualizar estadísticas semanales y mensuales
        this.actualizarEstadisticasPeriodo('semana', unaSemana);
        this.actualizarEstadisticasPeriodo('mes', unMes);

        // Actualizar gráficos de distribución
        this.actualizarGraficosPorTipo();
    }

    parseFecha(fechaStr) {
        const [dia, mes, año] = fechaStr.split('/');
        return `${año}-${mes}-${dia}`;
    }

    contarPorPropiedad(propiedad) {
        return this.comunicaciones.reduce((acc, curr) => {
            acc[curr[propiedad]] = (acc[curr[propiedad]] || 0) + 1;
            return acc;
        }, {});
    }

    actualizarEstadisticasPeriodo(tipo, duracion) {
        const hoy = new Date();
        const periodos = [3, 2, 1];

        periodos.forEach(n => {
            const inicio = new Date(hoy - (n * duracion));
            const fin = new Date(hoy - ((n - 1) * duracion));
            
            const comunicacionesPeriodo = this.comunicaciones.filter(c => {
                const fecha = new Date(this.parseFecha(c.fecha));
                return fecha >= inicio && fecha < fin;
            });

            const total = comunicacionesPeriodo.length;
            const recibidas = comunicacionesPeriodo.filter(c => c.estado === 'recibido').length;
            const rechazadas = comunicacionesPeriodo.filter(c => c.estado === 'rechazado').length;
            const invalidas = comunicacionesPeriodo.filter(c => c.estado === 'invalido').length;

            // Actualizar tabla
            document.getElementById(`${tipo}${n}Total`).textContent = total;
            document.getElementById(`${tipo}${n}Recibidas`).textContent = recibidas;
            document.getElementById(`${tipo}${n}Rechazadas`).textContent = rechazadas;
            document.getElementById(`${tipo}${n}Invalidas`).textContent = invalidas;

            // Actualizar gráfico
            const grafico = tipo === 'semana' ? this.graficoSemanal : this.graficoMensual;
            grafico.data.datasets[0].data[3-n] = recibidas;
            grafico.data.datasets[1].data[3-n] = rechazadas;
            grafico.data.datasets[2].data[3-n] = invalidas;
        });

        if (tipo === 'semana') {
            this.graficoSemanal.update();
        } else {
            this.graficoMensual.update();
        }
    }

    actualizarGraficosPorTipo() {
        const conteoMedios = this.contarPorPropiedad('medio');
        const conteoEstados = this.contarPorPropiedad('estado');

        this.graficoPorMedio.data.datasets[0].data = [
            conteoMedios['WhatsApp'] || 0,
            conteoMedios['Llamada'] || 0,
            conteoMedios['SMS'] || 0
        ];
        this.graficoPorMedio.update();

        this.graficoPorEstado.data.datasets[0].data = [
            conteoEstados['recibido'] || 0,
            conteoEstados['rechazado'] || 0,
            conteoEstados['invalido'] || 0
        ];
        this.graficoPorEstado.update();
    }
}

// Inicializar la UI y las estadísticas cuando el DOM esté listo
document.addEventListener('DOMContentLoaded', () => {
    try {
        window.comunicacionesUI = new ComunicacionesUI();
        window.estadisticas = new EstadisticasComunicaciones(comunicaciones);
    } catch (error) {
        console.error('Error al inicializar la interfaz:', error);
    }
}); 