// Función para generar una fecha aleatoria con peso hacia fechas recientes
function generarFechaAleatoria() {
    const hoy = new Date();
    let inicio, fin;
    
    // 40% de probabilidad de generar fechas en los últimos 3 meses
    if (Math.random() < 0.4) {
        inicio = new Date(hoy);
        inicio.setMonth(hoy.getMonth() - 3);
        fin = hoy;
    }
    // 30% de probabilidad de generar fechas en el último año
    else if (Math.random() < 0.7) {
        inicio = new Date(hoy);
        inicio.setFullYear(hoy.getFullYear() - 1);
        fin = hoy;
    }
    // 30% de probabilidad de generar fechas históricas
    else {
        inicio = new Date(2017, 0, 1);
        fin = new Date(2025, 11, 31);
    }

    const fechaAleatoria = new Date(inicio.getTime() + Math.random() * (fin.getTime() - inicio.getTime()));
    
    // Formatear la fecha para asegurar dos dígitos en día y mes
    const dia = fechaAleatoria.getDate().toString().padStart(2, '0');
    const mes = (fechaAleatoria.getMonth() + 1).toString().padStart(2, '0');
    const año = fechaAleatoria.getFullYear();
    
    return `${dia}/${mes}/${año}`;
}

// Función para generar un número de papeleta aleatorio
function generarPapeleta() {
    return 'P-' + Math.floor(Math.random() * 900000 + 100000);
}

// Tipos de papeletas con sus códigos y descripciones
const TIPOS_PAPELETA = {
    'V-01': 'Exceso de velocidad',
    'S-02': 'No respetar señal de tránsito',
    'E-03': 'Estacionamiento indebido',
    'L-04': 'Licencia vencida',
    'T-05': 'Transitar sin placa',
    'C-06': 'Conducir sin cinturón',
    'M-07': 'Uso de celular al conducir',
    'P-08': 'No respetar paso peatonal'
};

// Función para generar un código de papeleta aleatorio
function generarCodigoPapeleta() {
    const codigos = Object.keys(TIPOS_PAPELETA);
    return codigos[Math.floor(Math.random() * codigos.length)];
}

// Función para generar una placa aleatoria
function generarPlaca() {
    const letras = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    const placa = Array(3).fill().map(() => letras[Math.floor(Math.random() * letras.length)]).join('') +
                 '-' + Math.floor(Math.random() * 900 + 100);
    return placa;
}

// Función para generar un monto aleatorio
function generarMonto() {
    return (Math.floor(Math.random() * 90000) + 10000) / 100;
}

// Variables globales para almacenar datos
let registrosMultas = [];
let graficoPorMes;
let graficoPorTipo;

// Función para generar un estado de cobro aleatorio
function generarEstadoCobro() {
    return Math.random() < 0.6 ? 'Sí' : 'No'; // 60% de probabilidad de estar cobrado
}

// Función para generar una fila de datos
function generarFila() {
    const codigoPapeleta = generarCodigoPapeleta();
    return {
        fecha: generarFechaAleatoria(),
        papeleta: generarPapeleta(),
        codigoPapeleta: codigoPapeleta,
        tipoPapeleta: TIPOS_PAPELETA[codigoPapeleta],
        placa: generarPlaca(),
        monto: generarMonto(),
        cobrado: generarEstadoCobro()
    };
}

// Inicializar paginación
const tablaPagination = new TablePagination('tablaMultas');

// Función para formatear números con separadores de miles
function formatearNumero(numero) {
    return numero.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
}

// Función para actualizar la tabla con los datos
function actualizarTabla(multas) {
    const tabla = document.getElementById('tablaMultas');
    tabla.innerHTML = '';

    multas.forEach(multa => {
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>${multa.fecha}</td>
            <td>${multa.papeleta}</td>
            <td>${multa.codigoPapeleta} - ${multa.tipoPapeleta}</td>
            <td>${multa.placa}</td>
            <td>S/ ${formatearNumero(multa.monto.toFixed(2))}</td>
            <td class="estado-cobro ${multa.cobrado === 'Sí' ? 'cobrado' : 'no-cobrado'}">${multa.cobrado}</td>
        `;
        tabla.appendChild(row);
    });

    // Actualizar paginación con las nuevas filas
    tablaPagination.updateRows(tabla.getElementsByTagName('tr'));
}

// Función para obtener estadísticas por período
function obtenerEstadisticasPorPeriodo(multas, periodoEnDias) {
    const hoy = new Date();
    const inicio = new Date(hoy);
    inicio.setDate(hoy.getDate() - periodoEnDias);
    
    const multasDelPeriodo = multas.filter(multa => {
        const fechaMulta = new Date(multa.fecha.split('/').reverse().join('-'));
        return fechaMulta >= inicio && fechaMulta <= hoy;
    });

    const montoTotal = multasDelPeriodo.reduce((sum, multa) => sum + multa.monto, 0);
    
    return {
        total: multasDelPeriodo.length,
        monto: montoTotal,
        promedio: multasDelPeriodo.length ? montoTotal / multasDelPeriodo.length : 0
    };
}

// Función para actualizar las estadísticas comparativas
function actualizarEstadisticasComparativas(multas) {
    // Estadísticas de las últimas tres semanas
    const semana1 = obtenerEstadisticasPorPeriodo(multas, 7);
    const semana2 = obtenerEstadisticasPorPeriodo(multas, 14);
    const semana3 = obtenerEstadisticasPorPeriodo(multas, 21);

    // Estadísticas de los últimos tres meses
    const mes1 = obtenerEstadisticasPorPeriodo(multas, 30);
    const mes2 = obtenerEstadisticasPorPeriodo(multas, 60);
    const mes3 = obtenerEstadisticasPorPeriodo(multas, 90);

    // Crear gráfico comparativo de semanas
    const canvasSemanas = document.getElementById('comparativoSemanas');
    if (canvasSemanas.chart) {
        canvasSemanas.chart.destroy();
    }
    canvasSemanas.chart = new Chart(canvasSemanas, {
        type: 'bar',
        data: {
            labels: ['Última Semana', 'Hace 2 Semanas', 'Hace 3 Semanas'],
            datasets: [
                {
                    label: 'Cantidad de Papeletas',
                    data: [semana1.total, semana2.total - semana1.total, semana3.total - semana2.total],
                    backgroundColor: '#3f51b5',
                    yAxisID: 'y'
                },
                {
                    label: 'Monto Total (S/)',
                    data: [semana1.monto, semana2.monto - semana1.monto, semana3.monto - semana2.monto],
                    backgroundColor: '#f50057',
                    yAxisID: 'y1'
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                y: {
                    type: 'linear',
                    position: 'left',
                    title: {
                        display: true,
                        text: 'Cantidad de Papeletas'
                    }
                },
                y1: {
                    type: 'linear',
                    position: 'right',
                    title: {
                        display: true,
                        text: 'Monto Total (S/)'
                    },
                    grid: {
                        drawOnChartArea: false
                    }
                }
            },
            plugins: {
                title: {
                    display: true,
                    text: 'Comparativo de las Últimas 3 Semanas'
                }
            }
        }
    });

    // Crear gráfico comparativo de meses
    const canvasMeses = document.getElementById('comparativoMeses');
    if (canvasMeses.chart) {
        canvasMeses.chart.destroy();
    }
    canvasMeses.chart = new Chart(canvasMeses, {
        type: 'bar',
        data: {
            labels: ['Último Mes', 'Hace 2 Meses', 'Hace 3 Meses'],
            datasets: [
                {
                    label: 'Cantidad de Papeletas',
                    data: [mes1.total, mes2.total - mes1.total, mes3.total - mes2.total],
                    backgroundColor: '#4caf50',
                    yAxisID: 'y'
                },
                {
                    label: 'Monto Total (S/)',
                    data: [mes1.monto, mes2.monto - mes1.monto, mes3.monto - mes2.monto],
                    backgroundColor: '#ff9800',
                    yAxisID: 'y1'
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                y: {
                    type: 'linear',
                    position: 'left',
                    title: {
                        display: true,
                        text: 'Cantidad de Papeletas'
                    }
                },
                y1: {
                    type: 'linear',
                    position: 'right',
                    title: {
                        display: true,
                        text: 'Monto Total (S/)'
                    },
                    grid: {
                        drawOnChartArea: false
                    }
                }
            },
            plugins: {
                title: {
                    display: true,
                    text: 'Comparativo de los Últimos 3 Meses'
                }
            }
        }
    });
}

// Función para actualizar las estadísticas
function actualizarEstadisticas(multas) {
    document.getElementById('totalMultas').textContent = formatearNumero(multas.length);
    
    const montoTotal = multas.reduce((sum, multa) => sum + multa.monto, 0);
    document.getElementById('montoTotal').textContent = `S/ ${formatearNumero(montoTotal.toFixed(2))}`;
    
    const promedioMulta = montoTotal / multas.length || 0;
    document.getElementById('promedioMulta').textContent = `S/ ${formatearNumero(promedioMulta.toFixed(2))}`;
    
    const hoy = new Date();
    const multasDelMes = multas.filter(multa => {
        const fechaMulta = new Date(multa.fecha.split('/').reverse().join('-'));
        return fechaMulta.getMonth() === hoy.getMonth() && 
               fechaMulta.getFullYear() === hoy.getFullYear();
    });
    document.getElementById('multasMes').textContent = formatearNumero(multasDelMes.length);

    // Agregar estadísticas de cobro
    const multasCobradas = multas.filter(multa => multa.cobrado === 'Sí').length;
    const porcentajeCobrado = ((multasCobradas / multas.length) * 100).toFixed(1);
    document.getElementById('multasCobradas').textContent = `${porcentajeCobrado}%`;

    // Calcular y mostrar la distribución de montos
    const montoEntidad = montoTotal * 0.4;
    const montoAgencia = montoTotal * 0.6;
    document.getElementById('montoEntidad').textContent = `S/ ${formatearNumero(montoEntidad.toFixed(2))}`;
    document.getElementById('montoAgencia').textContent = `S/ ${formatearNumero(montoAgencia.toFixed(2))}`;
}

// Función para actualizar los gráficos
function actualizarGraficos(multas) {
    // Gráfico de multas por mes/año
    const multasPorMesAño = {};
    multas.forEach(multa => {
        const fecha = new Date(multa.fecha.split('/').reverse().join('-'));
        const mesAño = fecha.toLocaleString('es-ES', { month: 'long', year: 'numeric' });
        if (!multasPorMesAño[mesAño]) {
            multasPorMesAño[mesAño] = {
                total: 0,
                cobrado: 0,
                noCobrado: 0
            };
        }
        multasPorMesAño[mesAño].total++;
        if (multa.cobrado === 'Sí') {
            multasPorMesAño[mesAño].cobrado++;
        } else {
            multasPorMesAño[mesAño].noCobrado++;
        }
    });

    const canvasMes = document.getElementById('multasPorMes');
    if (canvasMes && canvasMes.chart) {
        canvasMes.chart.destroy();
    }
    if (canvasMes) {
        const labels = Object.keys(multasPorMesAño).slice(-12);
        canvasMes.chart = new Chart(canvasMes, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [
                    {
                        label: 'Cobradas',
                        data: labels.map(mes => multasPorMesAño[mes].cobrado),
                        backgroundColor: 'rgba(76, 175, 80, 0.7)',
                        stack: 'Stack 0'
                    },
                    {
                        label: 'No Cobradas',
                        data: labels.map(mes => multasPorMesAño[mes].noCobrado),
                        backgroundColor: 'rgba(244, 67, 54, 0.7)',
                        stack: 'Stack 0'
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: 'Papeletas por Mes/Año'
                    }
                },
                scales: {
                    x: {
                        stacked: true
                    },
                    y: {
                        stacked: true,
                        title: {
                            display: true,
                            text: 'Cantidad de Papeletas'
                        }
                    }
                }
            }
        });
    }

    // Gráfico de estado de cobro
    const canvasEstadoCobro = document.getElementById('estadoCobro');
    if (canvasEstadoCobro && canvasEstadoCobro.chart) {
        canvasEstadoCobro.chart.destroy();
    }
    if (canvasEstadoCobro) {
        const estadisticasCobro = {
            cobrado: multas.filter(m => m.cobrado === 'Sí').length,
            noCobrado: multas.filter(m => m.cobrado === 'No').length
        };

        canvasEstadoCobro.chart = new Chart(canvasEstadoCobro, {
            type: 'doughnut',
            data: {
                labels: ['Cobradas', 'No Cobradas'],
                datasets: [{
                    data: [estadisticasCobro.cobrado, estadisticasCobro.noCobrado],
                    backgroundColor: [
                        'rgba(76, 175, 80, 0.7)',  // Verde para cobradas
                        'rgba(244, 67, 54, 0.7)'   // Rojo para no cobradas
                    ],
                    borderColor: [
                        'rgba(76, 175, 80, 1)',
                        'rgba(244, 67, 54, 1)'
                    ],
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: 'Estado de Cobro de Papeletas'
                    },
                    legend: {
                        position: 'bottom'
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                const total = estadisticasCobro.cobrado + estadisticasCobro.noCobrado;
                                const porcentaje = ((context.raw / total) * 100).toFixed(1);
                                return `${context.label}: ${context.raw} (${porcentaje}%)`;
                            }
                        }
                    }
                }
            }
        });
    }

    // Mantener el gráfico de tipos de papeleta existente
    const papeletasPorTipo = {};
    multas.forEach(multa => {
        papeletasPorTipo[multa.tipoPapeleta] = (papeletasPorTipo[multa.tipoPapeleta] || 0) + 1;
    });

    const canvasTipo = document.getElementById('multasPorTipo');
    if (canvasTipo && canvasTipo.chart) {
        canvasTipo.chart.destroy();
    }
    if (canvasTipo) {
        canvasTipo.chart = new Chart(canvasTipo, {
            type: 'pie',
            data: {
                labels: Object.keys(papeletasPorTipo),
                datasets: [{
                    data: Object.values(papeletasPorTipo),
                    backgroundColor: [
                        '#3f51b5', // Azul
                        '#f50057', // Rosa
                        '#4caf50', // Verde
                        '#ff9800', // Naranja
                        '#9c27b0', // Morado
                        '#00bcd4', // Cyan
                        '#ff5722', // Naranja oscuro
                        '#795548'  // Marrón
                    ]
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'right',
                        labels: {
                            font: {
                                size: 12
                            },
                            padding: 20
                        }
                    },
                    title: {
                        display: true,
                        text: 'Distribución por Tipo de Papeleta'
                    }
                }
            }
        });
    }

    // Actualizar estadísticas comparativas si existen los elementos
    if (document.getElementById('comparativoSemanas') && document.getElementById('comparativoMeses')) {
        actualizarEstadisticasComparativas(multas);
    }
}

// Función para generar datos de prueba garantizando datos recientes
function generarDatosDePrueba(cantidad) {
    const datos = [];
    const hoy = new Date();

    // Asegurar datos para las últimas 3 semanas
    for (let i = 0; i < 15; i++) { // 5 registros por semana
        const fecha = new Date(hoy);
        fecha.setDate(hoy.getDate() - Math.floor(Math.random() * 21)); // Últimos 21 días
        const dia = fecha.getDate().toString().padStart(2, '0');
        const mes = (fecha.getMonth() + 1).toString().padStart(2, '0');
        const año = fecha.getFullYear();
        datos.push({
            fecha: `${dia}/${mes}/${año}`,
            ...generarDatosRegistro()
        });
    }

    // Asegurar datos para los últimos 3 meses
    for (let i = 0; i < 30; i++) { // 10 registros por mes
        const fecha = new Date(hoy);
        fecha.setDate(hoy.getDate() - Math.floor(Math.random() * 90)); // Últimos 90 días
        const dia = fecha.getDate().toString().padStart(2, '0');
        const mes = (fecha.getMonth() + 1).toString().padStart(2, '0');
        const año = fecha.getFullYear();
        datos.push({
            fecha: `${dia}/${mes}/${año}`,
            ...generarDatosRegistro()
        });
    }

    // Generar el resto de datos históricos
    const restantes = cantidad - 45; // 45 = 15 (semanas) + 30 (meses)
    for (let i = 0; i < restantes; i++) {
        datos.push(generarFila());
    }

    return datos;
}

// Función auxiliar para generar los datos de un registro sin la fecha
function generarDatosRegistro() {
    return {
        papeleta: generarPapeleta(),
        codigoPapeleta: generarCodigoPapeleta(),
        placa: generarPlaca(),
        monto: generarMonto(),
        cobrado: generarEstadoCobro()
    };
}

// Modificar el evento load para usar la nueva función de generación de datos
window.addEventListener('load', () => {
    // Generar 100 registros con datos garantizados para períodos recientes
    registrosMultas = generarDatosDePrueba(100);

    // Ordenar registros por fecha (más reciente primero)
    registrosMultas.sort((a, b) => {
        const fechaA = new Date(a.fecha.split('/').reverse().join('-'));
        const fechaB = new Date(b.fecha.split('/').reverse().join('-'));
        return fechaB - fechaA;
    });

    // Actualizar tabla, estadísticas y gráficos
    actualizarTabla(registrosMultas);
    actualizarEstadisticas(registrosMultas);
    actualizarGraficos(registrosMultas);
}); 