// Función para generar una fecha aleatoria con peso hacia fechas recientes
function generarFechaAleatoria() {
    const hoy = new Date();
    let inicio, fin;
    
    // 40% de probabilidad de generar fechas en los últimos 3 meses
    if (Math.random() < 0.4) {
        inicio = new Date(hoy);
        inicio.setMonth(hoy.getMonth() - 3);
        fin = hoy;
    }
    // 30% de probabilidad de generar fechas en el último año
    else if (Math.random() < 0.7) {
        inicio = new Date(hoy);
        inicio.setFullYear(hoy.getFullYear() - 1);
        fin = hoy;
    }
    // 30% de probabilidad de generar fechas históricas
    else {
        inicio = new Date(2023, 0, 1);
        fin = new Date(2024, 11, 31);
    }

    const fechaAleatoria = new Date(inicio.getTime() + Math.random() * (fin.getTime() - inicio.getTime()));
    
    // Formatear la fecha como DD/MM/YYYY
    const dia = fechaAleatoria.getDate().toString().padStart(2, '0');
    const mes = (fechaAleatoria.getMonth() + 1).toString().padStart(2, '0');
    const año = fechaAleatoria.getFullYear();
    
    return `${dia}/${mes}/${año}`;
}

// Función para generar un número de papeleta aleatorio
function generarPapeleta() {
    return 'P-' + Math.floor(Math.random() * 900000 + 100000);
}

// Función para generar un código de papeleta aleatorio
function generarCodigoPapeleta() {
    const letras = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    const numeros = '0123456789';
    let codigo = '';
    
    // Generar 3 letras aleatorias
    for (let i = 0; i < 3; i++) {
        codigo += letras.charAt(Math.floor(Math.random() * letras.length));
    }
    
    // Agregar guión
    codigo += '-';
    
    // Generar 4 números aleatorios
    for (let i = 0; i < 4; i++) {
        codigo += numeros.charAt(Math.floor(Math.random() * numeros.length));
    }
    
    return codigo;
}

// Función para generar una placa aleatoria
function generarPlaca() {
    const letras = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    const placa = Array(3).fill().map(() => letras[Math.floor(Math.random() * letras.length)]).join('') +
                 '-' + Math.floor(Math.random() * 900 + 100);
    return placa;
}

// Función para generar un monto aleatorio
function generarMonto() {
    return (Math.floor(Math.random() * 90000) + 10000) / 100;
}

// Función para generar un método de pago aleatorio
function generarMetodoPago() {
    const metodos = ['Efectivo 💵', 'Visa 💳', 'Mastercard 💳', 'Plin 📱', 'Yape 📱'];
    return metodos[Math.floor(Math.random() * metodos.length)];
}

// Función para generar un estado de cobro aleatorio
function generarEstadoCobro() {
    return Math.random() < 0.6 ? 'Sí' : 'No'; // 60% de probabilidad de estar cobrado
}

// Variables globales para almacenar datos
let registrosCobranzas = [];

// Función para generar una fila de datos
function generarFila() {
    return {
        fecha: generarFechaAleatoria(),
        papeleta: generarPapeleta(),
        codigoPapeleta: generarCodigoPapeleta(),
        placa: generarPlaca(),
        monto: generarMonto(),
        metodoPago: generarMetodoPago(),
        cobrado: generarEstadoCobro()
    };
}

// Inicializar paginación
const tablaPagination = new TablePagination('tablaCobranzas');

// Función para formatear números con separadores de miles y decimales
function formatearNumero(numero) {
    return numero.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
}

// Función para formatear montos con separadores de miles y dos decimales
function formatearMonto(monto) {
    return `S/ ${formatearNumero(parseFloat(monto).toFixed(2))}`;
}

// Función para actualizar la tabla con los datos
function actualizarTabla(cobranzas) {
    const tabla = document.getElementById('tablaCobranzas');
    tabla.innerHTML = '';

    cobranzas.forEach(cobranza => {
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>${cobranza.fecha}</td>
            <td>${cobranza.papeleta}</td>
            <td>${cobranza.codigoPapeleta}</td>
            <td>${cobranza.placa}</td>
            <td>${formatearMonto(cobranza.monto)}</td>
            <td>${cobranza.metodoPago}</td>
        `;
        tabla.appendChild(row);
    });

    // Actualizar paginación con las nuevas filas
    tablaPagination.updateRows(tabla.getElementsByTagName('tr'));
}

// Función para obtener estadísticas por período
function obtenerEstadisticasPeriodo(registros, diasInicio, diasFin) {
    const hoy = new Date();
    const inicio = new Date(hoy);
    const fin = new Date(hoy);
    inicio.setDate(hoy.getDate() - diasInicio);
    fin.setDate(hoy.getDate() - diasFin);
    
    const registrosPeriodo = registros.filter(registro => {
        const [dia, mes, año] = registro.fecha.split('/');
        const fecha = new Date(año, mes - 1, dia);
        return fecha >= fin && fecha < inicio;
    });

    return {
        total: registrosPeriodo.length,
        cobradas: registrosPeriodo.filter(r => r.cobrado === 'Sí').length,
        pendientes: registrosPeriodo.filter(r => r.cobrado === 'No').length
    };
}

// Función para actualizar estadísticas semanales
function actualizarEstadisticasSemanales(registros) {
    // Última semana
    const semana1 = obtenerEstadisticasPeriodo(registros, 0, 7);
    document.getElementById('semana1Total').textContent = formatearNumero(semana1.total);
    document.getElementById('semana1Cobradas').textContent = formatearNumero(semana1.cobradas);
    document.getElementById('semana1Pendientes').textContent = formatearNumero(semana1.pendientes);

    // Hace 2 semanas
    const semana2 = obtenerEstadisticasPeriodo(registros, 7, 14);
    document.getElementById('semana2Total').textContent = formatearNumero(semana2.total);
    document.getElementById('semana2Cobradas').textContent = formatearNumero(semana2.cobradas);
    document.getElementById('semana2Pendientes').textContent = formatearNumero(semana2.pendientes);

    // Hace 3 semanas
    const semana3 = obtenerEstadisticasPeriodo(registros, 14, 21);
    document.getElementById('semana3Total').textContent = formatearNumero(semana3.total);
    document.getElementById('semana3Cobradas').textContent = formatearNumero(semana3.cobradas);
    document.getElementById('semana3Pendientes').textContent = formatearNumero(semana3.pendientes);
}

// Función para actualizar estadísticas mensuales
function actualizarEstadisticasMensuales(registros) {
    // Último mes
    const mes1 = obtenerEstadisticasPeriodo(registros, 0, 30);
    document.getElementById('mes1Total').textContent = formatearNumero(mes1.total);
    document.getElementById('mes1Cobradas').textContent = formatearNumero(mes1.cobradas);
    document.getElementById('mes1Pendientes').textContent = formatearNumero(mes1.pendientes);

    // Hace 2 meses
    const mes2 = obtenerEstadisticasPeriodo(registros, 30, 60);
    document.getElementById('mes2Total').textContent = formatearNumero(mes2.total);
    document.getElementById('mes2Cobradas').textContent = formatearNumero(mes2.cobradas);
    document.getElementById('mes2Pendientes').textContent = formatearNumero(mes2.pendientes);

    // Hace 3 meses
    const mes3 = obtenerEstadisticasPeriodo(registros, 60, 90);
    document.getElementById('mes3Total').textContent = formatearNumero(mes3.total);
    document.getElementById('mes3Cobradas').textContent = formatearNumero(mes3.cobradas);
    document.getElementById('mes3Pendientes').textContent = formatearNumero(mes3.pendientes);
}

// Función para actualizar el gráfico semanal
function actualizarGraficoSemanal(registros) {
    const semana1 = obtenerEstadisticasPeriodo(registros, 0, 7);
    const semana2 = obtenerEstadisticasPeriodo(registros, 7, 14);
    const semana3 = obtenerEstadisticasPeriodo(registros, 14, 21);

    const canvas = document.getElementById('graficoSemanal');
    if (canvas.chart) {
        canvas.chart.destroy();
    }

    canvas.chart = new Chart(canvas, {
        type: 'bar',
        data: {
            labels: ['Última Semana', 'Hace 2 Semanas', 'Hace 3 Semanas'],
            datasets: [
                {
                    label: 'Total Papeletas',
                    data: [semana1.total, semana2.total, semana3.total],
                    backgroundColor: '#3f51b5',
                    borderColor: '#3f51b5',
                    borderWidth: 1
                },
                {
                    label: 'Cobradas',
                    data: [semana1.cobradas, semana2.cobradas, semana3.cobradas],
                    backgroundColor: '#4caf50',
                    borderColor: '#4caf50',
                    borderWidth: 1
                },
                {
                    label: 'Pendientes',
                    data: [semana1.pendientes, semana2.pendientes, semana3.pendientes],
                    backgroundColor: '#f50057',
                    borderColor: '#f50057',
                    borderWidth: 1
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                title: {
                    display: true,
                    text: 'Comparativo Semanal'
                },
                legend: {
                    position: 'bottom'
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    title: {
                        display: true,
                        text: 'Cantidad de Papeletas'
                    }
                }
            }
        }
    });
}

// Función para actualizar el gráfico mensual
function actualizarGraficoMensual(registros) {
    const mes1 = obtenerEstadisticasPeriodo(registros, 0, 30);
    const mes2 = obtenerEstadisticasPeriodo(registros, 30, 60);
    const mes3 = obtenerEstadisticasPeriodo(registros, 60, 90);

    const canvas = document.getElementById('graficoMensual');
    if (canvas.chart) {
        canvas.chart.destroy();
    }

    canvas.chart = new Chart(canvas, {
        type: 'bar',
        data: {
            labels: ['Último Mes', 'Hace 2 Meses', 'Hace 3 Meses'],
            datasets: [
                {
                    label: 'Total Papeletas',
                    data: [mes1.total, mes2.total, mes3.total],
                    backgroundColor: '#3f51b5',
                    borderColor: '#3f51b5',
                    borderWidth: 1
                },
                {
                    label: 'Cobradas',
                    data: [mes1.cobradas, mes2.cobradas, mes3.cobradas],
                    backgroundColor: '#4caf50',
                    borderColor: '#4caf50',
                    borderWidth: 1
                },
                {
                    label: 'Pendientes',
                    data: [mes1.pendientes, mes2.pendientes, mes3.pendientes],
                    backgroundColor: '#f50057',
                    borderColor: '#f50057',
                    borderWidth: 1
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                title: {
                    display: true,
                    text: 'Comparativo Mensual'
                },
                legend: {
                    position: 'bottom'
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    title: {
                        display: true,
                        text: 'Cantidad de Papeletas'
                    }
                }
            }
        }
    });
}

// Modificar la función de actualización principal para incluir los nuevos gráficos
function actualizarEstadisticas(registros) {
    // Actualizar estadísticas existentes
    document.getElementById('totalCobranzas').textContent = formatearNumero(registros.length);
    
    const montoTotal = registros.reduce((sum, registro) => sum + registro.monto, 0);
    document.getElementById('montoTotal').textContent = formatearMonto(montoTotal);
    
    // Calcular método más usado
    const metodosPago = registros.reduce((acc, registro) => {
        acc[registro.metodoPago] = (acc[registro.metodoPago] || 0) + 1;
        return acc;
    }, {});
    
    const metodoPrincipal = Object.entries(metodosPago)
        .sort((a, b) => b[1] - a[1])[0];
    document.getElementById('metodoPrincipal').textContent = metodoPrincipal ? metodoPrincipal[0] : '-';
    
    // Cobranzas de hoy
    const hoy = new Date().toLocaleDateString('es-ES');
    const cobranzasHoy = registros.filter(registro => registro.fecha === hoy).length;
    document.getElementById('cobranzasHoy').textContent = formatearNumero(cobranzasHoy);

    // Actualizar estadísticas comparativas y gráficos
    actualizarEstadisticasSemanales(registros);
    actualizarEstadisticasMensuales(registros);
    actualizarGraficoSemanal(registros);
    actualizarGraficoMensual(registros);
}

// Función para actualizar los gráficos
function actualizarGraficos(cobranzas) {
    // Gráfico de cobranzas por mes
    const cobranzasPorMes = {};
    cobranzas.forEach(cobranza => {
        const fecha = new Date(cobranza.fecha.split('/').reverse().join('-'));
        const mes = fecha.toLocaleString('es-ES', { month: 'long' });
        cobranzasPorMes[mes] = (cobranzasPorMes[mes] || 0) + cobranza.monto;
    });

    const canvasMes = document.getElementById('cobranzasPorMes');
    if (canvasMes.chart) {
        canvasMes.chart.destroy();
    }
    canvasMes.chart = new Chart(canvasMes, {
        type: 'bar',
        data: {
            labels: Object.keys(cobranzasPorMes),
            datasets: [{
                label: 'Monto Cobrado por Mes',
                data: Object.values(cobranzasPorMes),
                backgroundColor: '#3f51b5'
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            return formatearMonto(context.raw);
                        }
                    }
                }
            },
            scales: {
                y: {
                    ticks: {
                        callback: function(value) {
                            return formatearMonto(value);
                        }
                    }
                }
            }
        }
    });

    // Gráfico de cobranzas por método de pago
    const cobranzasPorMetodo = {};
    cobranzas.forEach(cobranza => {
        const metodoPago = cobranza.metodoPago.split(' ')[0]; // Obtener solo el nombre sin el emoji
        cobranzasPorMetodo[metodoPago] = (cobranzasPorMetodo[metodoPago] || 0) + cobranza.monto;
    });

    const canvasMetodo = document.getElementById('cobranzasPorMetodo');
    if (canvasMetodo.chart) {
        canvasMetodo.chart.destroy();
    }
    canvasMetodo.chart = new Chart(canvasMetodo, {
        type: 'pie',
        data: {
            labels: Object.keys(cobranzasPorMetodo),
            datasets: [{
                data: Object.values(cobranzasPorMetodo),
                backgroundColor: [
                    '#3f51b5',
                    '#f50057',
                    '#4caf50',
                    '#ff9800',
                    '#9c27b0'
                ]
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false
        }
    });
}

// Función para generar datos garantizando registros recientes
function generarDatosDePrueba(cantidad) {
    const datos = [];
    const hoy = new Date();

    // Asegurar datos para las últimas 3 semanas
    for (let i = 0; i < 15; i++) { // 5 registros por semana
        const fecha = new Date(hoy);
        fecha.setDate(hoy.getDate() - Math.floor(Math.random() * 21)); // Últimos 21 días
        const dia = fecha.getDate().toString().padStart(2, '0');
        const mes = (fecha.getMonth() + 1).toString().padStart(2, '0');
        const año = fecha.getFullYear();
        datos.push({
            fecha: `${dia}/${mes}/${año}`,
            papeleta: generarPapeleta(),
            codigoPapeleta: generarCodigoPapeleta(),
            placa: generarPlaca(),
            monto: generarMonto(),
            metodoPago: generarMetodoPago(),
            cobrado: generarEstadoCobro()
        });
    }

    // Asegurar datos para los últimos 3 meses
    for (let i = 0; i < 30; i++) { // 10 registros por mes
        const fecha = new Date(hoy);
        fecha.setDate(hoy.getDate() - Math.floor(Math.random() * 90)); // Últimos 90 días
        const dia = fecha.getDate().toString().padStart(2, '0');
        const mes = (fecha.getMonth() + 1).toString().padStart(2, '0');
        const año = fecha.getFullYear();
        datos.push({
            fecha: `${dia}/${mes}/${año}`,
            papeleta: generarPapeleta(),
            codigoPapeleta: generarCodigoPapeleta(),
            placa: generarPlaca(),
            monto: generarMonto(),
            metodoPago: generarMetodoPago(),
            cobrado: generarEstadoCobro()
        });
    }

    // Generar el resto de datos históricos
    const restantes = cantidad - 45; // 45 = 15 (semanas) + 30 (meses)
    for (let i = 0; i < restantes; i++) {
        datos.push(generarFila());
    }

    return datos;
}

// Modificar el evento load para usar la nueva función de generación de datos
window.addEventListener('load', () => {
    // Generar 100 registros con datos garantizados para períodos recientes
    registrosCobranzas = generarDatosDePrueba(100);

    // Ordenar registros por fecha (más reciente primero)
    registrosCobranzas.sort((a, b) => {
        const [diaA, mesA, añoA] = a.fecha.split('/');
        const [diaB, mesB, añoB] = b.fecha.split('/');
        const fechaA = new Date(añoA, mesA - 1, diaA);
        const fechaB = new Date(añoB, mesB - 1, diaB);
        return fechaB - fechaA;
    });

    // Actualizar tabla, estadísticas y gráficos
    actualizarTabla(registrosCobranzas);
    actualizarEstadisticas(registrosCobranzas);
    actualizarGraficos(registrosCobranzas);
}); 