// Datos de ejemplo de los cobradores
const cobradores = [
    {
        nombre: "José Gutiérrez",
        zona: "Zona Norte",
        iniciales: "JG",
        total: "S/ 15,420.50",
        cobranzas: 45,
        efectividad: "92%",
        telefono: "+51 987 654 321",
        email: "jose.gutierrez@cobranzas.com",
        estadisticas_semana: [
            { fecha: '2024-03-11', cobranzas: 8, monto: 2850.00, efectividad: 90 },
            { fecha: '2024-03-12', cobranzas: 7, monto: 2420.50, efectividad: 95 },
            { fecha: '2024-03-13', cobranzas: 6, monto: 1980.00, efectividad: 88 },
            { fecha: '2024-03-14', cobranzas: 8, monto: 2640.00, efectividad: 94 },
            { fecha: '2024-03-15', cobranzas: 5, monto: 1750.00, efectividad: 92 },
            { fecha: '2024-03-16', cobranzas: 6, monto: 2180.00, efectividad: 91 },
            { fecha: '2024-03-17', cobranzas: 5, monto: 1600.00, efectividad: 93 }
        ]
    },
    {
        nombre: "María Rodríguez",
        zona: "Zona Sur",
        iniciales: "MR",
        total: "S/ 12,850.75",
        cobranzas: 38,
        efectividad: "88%",
        telefono: "+51 956 789 123",
        email: "maria.rodriguez@cobranzas.com",
        estadisticas_semana: [
            { fecha: '2024-03-11', cobranzas: 6, monto: 2100.00, efectividad: 85 },
            { fecha: '2024-03-12', cobranzas: 5, monto: 1850.75, efectividad: 87 },
            { fecha: '2024-03-13', cobranzas: 7, monto: 2300.00, efectividad: 89 },
            { fecha: '2024-03-14', cobranzas: 5, monto: 1750.00, efectividad: 86 },
            { fecha: '2024-03-15', cobranzas: 6, monto: 1950.00, efectividad: 88 },
            { fecha: '2024-03-16', cobranzas: 4, monto: 1500.00, efectividad: 90 },
            { fecha: '2024-03-17', cobranzas: 5, monto: 1400.00, efectividad: 91 }
        ]
    },
    {
        nombre: "Carlos Pérez",
        zona: "Zona Este",
        iniciales: "CP",
        total: "S/ 18,730.25",
        cobranzas: 52,
        efectividad: "95%",
        telefono: "+51 945 678 912",
        email: "carlos.perez@cobranzas.com",
        estadisticas_semana: [
            { fecha: '2024-03-11', cobranzas: 8, monto: 2800.00, efectividad: 94 },
            { fecha: '2024-03-12', cobranzas: 9, monto: 3100.25, efectividad: 96 },
            { fecha: '2024-03-13', cobranzas: 7, monto: 2450.00, efectividad: 95 },
            { fecha: '2024-03-14', cobranzas: 8, monto: 2780.00, efectividad: 93 },
            { fecha: '2024-03-15', cobranzas: 7, monto: 2500.00, efectividad: 95 },
            { fecha: '2024-03-16', cobranzas: 6, monto: 2300.00, efectividad: 97 },
            { fecha: '2024-03-17', cobranzas: 7, monto: 2800.00, efectividad: 94 }
        ]
    },
    {
        nombre: "Ana Torres",
        zona: "Zona Oeste",
        iniciales: "AT",
        total: "S/ 14,280.00",
        cobranzas: 41,
        efectividad: "87%",
        telefono: "+51 934 567 891",
        email: "ana.torres@cobranzas.com",
        estadisticas_semana: [
            { fecha: '2024-03-11', cobranzas: 6, monto: 2100.00, efectividad: 86 },
            { fecha: '2024-03-12', cobranzas: 5, monto: 1750.00, efectividad: 88 },
            { fecha: '2024-03-13', cobranzas: 7, monto: 2430.00, efectividad: 85 },
            { fecha: '2024-03-14', cobranzas: 6, monto: 2100.00, efectividad: 87 },
            { fecha: '2024-03-15', cobranzas: 5, monto: 1800.00, efectividad: 89 },
            { fecha: '2024-03-16', cobranzas: 6, monto: 2200.00, efectividad: 86 },
            { fecha: '2024-03-17', cobranzas: 6, monto: 1900.00, efectividad: 88 }
        ]
    },
    {
        nombre: "Luis Mendoza",
        zona: "Zona Norte",
        iniciales: "LM",
        total: "S/ 16,890.30",
        cobranzas: 49,
        efectividad: "91%",
        telefono: "+51 923 456 789",
        email: "luis.mendoza@cobranzas.com",
        estadisticas_semana: [
            { fecha: '2024-03-11', cobranzas: 7, monto: 2450.00, efectividad: 90 },
            { fecha: '2024-03-12', cobranzas: 8, monto: 2780.30, efectividad: 92 },
            { fecha: '2024-03-13', cobranzas: 7, monto: 2400.00, efectividad: 91 },
            { fecha: '2024-03-14', cobranzas: 6, monto: 2160.00, efectividad: 89 },
            { fecha: '2024-03-15', cobranzas: 7, monto: 2500.00, efectividad: 93 },
            { fecha: '2024-03-16', cobranzas: 8, monto: 2800.00, efectividad: 90 },
            { fecha: '2024-03-17', cobranzas: 6, monto: 1800.00, efectividad: 92 }
        ]
    },
    {
        nombre: "Carmen Vega",
        zona: "Zona Sur",
        iniciales: "CV",
        total: "S/ 11,450.80",
        cobranzas: 35,
        efectividad: "85%",
        telefono: "+51 912 345 678",
        email: "carmen.vega@cobranzas.com",
        estadisticas_semana: [
            { fecha: '2024-03-11', cobranzas: 5, monto: 1750.00, efectividad: 84 },
            { fecha: '2024-03-12', cobranzas: 4, monto: 1400.80, efectividad: 86 },
            { fecha: '2024-03-13', cobranzas: 6, monto: 2100.00, efectividad: 83 },
            { fecha: '2024-03-14', cobranzas: 5, monto: 1650.00, efectividad: 85 },
            { fecha: '2024-03-15', cobranzas: 4, monto: 1450.00, efectividad: 87 },
            { fecha: '2024-03-16', cobranzas: 6, monto: 1900.00, efectividad: 84 },
            { fecha: '2024-03-17', cobranzas: 5, monto: 1200.00, efectividad: 86 }
        ]
    },
    {
        nombre: "Roberto Silva",
        zona: "Zona Este",
        iniciales: "RS",
        total: "S/ 16,240.60",
        cobranzas: 47,
        efectividad: "93%",
        telefono: "+51 901 234 567",
        email: "roberto.silva@cobranzas.com",
        estadisticas_semana: [
            { fecha: '2024-03-11', cobranzas: 7, monto: 2450.00, efectividad: 92 },
            { fecha: '2024-03-12', cobranzas: 8, monto: 2780.60, efectividad: 94 },
            { fecha: '2024-03-13', cobranzas: 6, monto: 2100.00, efectividad: 91 },
            { fecha: '2024-03-14', cobranzas: 7, monto: 2310.00, efectividad: 93 },
            { fecha: '2024-03-15', cobranzas: 6, monto: 2200.00, efectividad: 95 },
            { fecha: '2024-03-16', cobranzas: 7, monto: 2500.00, efectividad: 92 },
            { fecha: '2024-03-17', cobranzas: 6, monto: 1900.00, efectividad: 94 }
        ]
    },
    {
        nombre: "Patricia Luna",
        zona: "Zona Oeste",
        iniciales: "PL",
        total: "S/ 15,180.90",
        cobranzas: 43,
        efectividad: "89%",
        telefono: "+51 989 012 345",
        email: "patricia.luna@cobranzas.com",
        estadisticas_semana: [
            { fecha: '2024-03-11', cobranzas: 6, monto: 2100.00, efectividad: 88 },
            { fecha: '2024-03-12', cobranzas: 7, monto: 2430.90, efectividad: 90 },
            { fecha: '2024-03-13', cobranzas: 6, monto: 2100.00, efectividad: 87 },
            { fecha: '2024-03-14', cobranzas: 5, monto: 1750.00, efectividad: 89 },
            { fecha: '2024-03-15', cobranzas: 7, monto: 2400.00, efectividad: 91 },
            { fecha: '2024-03-16', cobranzas: 6, monto: 2200.00, efectividad: 88 },
            { fecha: '2024-03-17', cobranzas: 6, monto: 2200.00, efectividad: 90 }
        ]
    },
    {
        nombre: "Miguel Ríos",
        zona: "Zona Norte",
        iniciales: "MR",
        total: "S/ 17,920.40",
        cobranzas: 51,
        efectividad: "94%",
        telefono: "+51 978 901 234",
        email: "miguel.rios@cobranzas.com",
        estadisticas_semana: [
            { fecha: '2024-03-11', cobranzas: 8, monto: 2800.00, efectividad: 93 },
            { fecha: '2024-03-12', cobranzas: 7, monto: 2450.40, efectividad: 95 },
            { fecha: '2024-03-13', cobranzas: 8, monto: 2780.00, efectividad: 92 },
            { fecha: '2024-03-14', cobranzas: 7, monto: 2470.00, efectividad: 94 },
            { fecha: '2024-03-15', cobranzas: 8, monto: 2820.00, efectividad: 96 },
            { fecha: '2024-03-16', cobranzas: 7, monto: 2400.00, efectividad: 93 },
            { fecha: '2024-03-17', cobranzas: 6, monto: 2200.00, efectividad: 95 }
        ]
    },
    {
        nombre: "Diana Castro",
        zona: "Zona Sur",
        iniciales: "DC",
        total: "S/ 13,560.70",
        cobranzas: 39,
        efectividad: "86%",
        telefono: "+51 967 890 123",
        email: "diana.castro@cobranzas.com",
        estadisticas_semana: [
            { fecha: '2024-03-11', cobranzas: 6, monto: 2100.00, efectividad: 85 },
            { fecha: '2024-03-12', cobranzas: 5, monto: 1750.70, efectividad: 87 },
            { fecha: '2024-03-13', cobranzas: 6, monto: 2100.00, efectividad: 84 },
            { fecha: '2024-03-14', cobranzas: 5, monto: 1760.00, efectividad: 86 },
            { fecha: '2024-03-15', cobranzas: 6, monto: 2050.00, efectividad: 88 },
            { fecha: '2024-03-16', cobranzas: 5, monto: 1900.00, efectividad: 85 },
            { fecha: '2024-03-17', cobranzas: 6, monto: 1900.00, efectividad: 87 }
        ]
    }
];

// Clase para manejar la interfaz de usuario
class CobradoresUI {
    constructor() {
        // Referencias a elementos del DOM
        this.gridContainer = document.querySelector('.cobradores-grid');
        this.drawer = null;
        this.topAppBar = null;
        this.searchField = null;
        this.zoneSelect = null;
        this.charts = new Map(); // Almacena las referencias a los gráficos
    }

    // Inicializa todos los componentes de Material Design
    initializeMDC() {
        // Drawer
        const drawerEl = document.querySelector('.mdc-drawer');
        if (drawerEl) {
            this.drawer = new mdc.drawer.MDCDrawer(drawerEl);
        }

        // Top App Bar
        const topAppBarEl = document.querySelector('.mdc-top-app-bar');
        if (topAppBarEl) {
            this.topAppBar = new mdc.topAppBar.MDCTopAppBar(topAppBarEl);
            
            // Configurar interacción entre Top App Bar y Drawer
            if (this.drawer) {
                const contentEl = document.querySelector('.page-content');
                if (contentEl) {
                    this.topAppBar.setScrollTarget(contentEl);
                    this.topAppBar.listen('MDCTopAppBar:nav', () => {
                        this.drawer.open = !this.drawer.open;
                    });
                }
            }
        }

        // Campo de búsqueda
        const searchFieldEl = document.querySelector('.search-field');
        if (searchFieldEl) {
            this.searchField = new mdc.textField.MDCTextField(searchFieldEl);
            this.searchField.input.addEventListener('input', () => this.filterCobradores());
        }

        // Selector de zona
        const selectEl = document.querySelector('.mdc-select');
        if (selectEl) {
            this.zoneSelect = new mdc.select.MDCSelect(selectEl);
            this.zoneSelect.listen('MDCSelect:change', () => this.filterCobradores());
        }

        // Inicializar ripple en botones de iconos
        document.querySelectorAll('.mdc-icon-button').forEach(icon => {
            new mdc.ripple.MDCRipple(icon).unbounded = true;
        });
    }

    // Crea un gráfico de estadísticas del cobrador
    createChart(canvasId, estadisticas) {
        const canvas = document.getElementById(canvasId);
        if (!canvas || !estadisticas) return null;

        // Destruir gráfico existente si hay uno
        if (this.charts.has(canvasId)) {
            this.charts.get(canvasId).destroy();
        }

        const chart = new Chart(canvas, {
            type: 'bar',
            data: {
                labels: ['Llamadas', 'SMS', 'WhatsApp'],
                datasets: [
                    {
                        type: 'bar',
                        label: 'Contactos',
                        data: [estadisticas.llamadas, estadisticas.sms, estadisticas.whatsapp],
                        backgroundColor: [
                            'rgba(33, 150, 243, 0.7)',  // Azul para llamadas
                            'rgba(255, 152, 0, 0.7)',   // Naranja para SMS
                            'rgba(76, 175, 80, 0.7)'    // Verde para WhatsApp
                        ],
                        borderColor: [
                            'rgb(33, 150, 243)',
                            'rgb(255, 152, 0)',
                            'rgb(76, 175, 80)'
                        ],
                        borderWidth: 1,
                        order: 2
                    },
                    {
                        type: 'line',
                        label: 'Tasa de Respuesta',
                        data: [
                            (estadisticas.llamadas * 0.6),  // 60% tasa de respuesta llamadas
                            (estadisticas.sms * 0.4),       // 40% tasa de respuesta SMS
                            (estadisticas.whatsapp * 0.8)   // 80% tasa de respuesta WhatsApp
                        ],
                        borderColor: 'rgba(156, 39, 176, 0.8)',  // Morado para la línea
                        borderWidth: 2,
                        pointBackgroundColor: 'rgba(156, 39, 176, 1)',
                        pointRadius: 4,
                        fill: false,
                        tension: 0.4,
                        order: 1
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: true,
                        position: 'top',
                        labels: {
                            usePointStyle: true,
                            padding: 15
                        }
                    },
                    tooltip: {
                        mode: 'index',
                        intersect: false,
                        callbacks: {
                            label: function(context) {
                                let label = context.dataset.label || '';
                                if (label) {
                                    label += ': ';
                                }
                                label += context.parsed.y.toFixed(0);
                                return label;
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            drawBorder: false,
                            color: 'rgba(0, 0, 0, 0.1)'
                        },
                        ticks: {
                            callback: function(value) {
                                return value.toFixed(0);
                            }
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        }
                    }
                },
                interaction: {
                    intersect: false,
                    mode: 'index'
                }
            }
        });

        this.charts.set(canvasId, chart);
        return chart;
    }

    // Crea una tarjeta de cobrador
    createCobradorCard(cobrador, index) {
        const card = document.createElement('div');
        card.className = 'mdc-card content-section';
        
        const iniciales = cobrador.nombre.split(' ').map(n => n[0]).join('');
        const canvasId = `grafico-${index}-${Date.now()}`;
        
        card.innerHTML = `
            <div class="cobrador-header">
                <div class="cobrador-avatar">${iniciales}</div>
                <div class="cobrador-info">
                    <h3 class="cobrador-name">${cobrador.nombre}</h3>
                    <p class="mdc-typography--body2">${cobrador.zona}</p>
                </div>
                <button class="mdc-icon-button material-icons" aria-label="Más opciones">more_vert</button>
            </div>
            <div class="cobrador-stats">
                <div class="stat-item">
                    <div class="stat-value">${cobrador.cobranzas}</div>
                    <div class="stat-label">Cobranzas</div>
                </div>
                <div class="stat-item">
                    <div class="stat-value">S/ ${cobrador.total.toLocaleString('es-PE', {
                        minimumFractionDigits: 2,
                        maximumFractionDigits: 2
                    })}</div>
                    <div class="stat-label">Total</div>
                </div>
                <div class="stat-item">
                    <div class="stat-value">${cobrador.efectividad}</div>
                    <div class="stat-label">Efectividad</div>
                </div>
            </div>
            <div class="success-rate">
                <div class="success-rate-fill" style="width: ${cobrador.efectividad}%"></div>
            </div>
            <div class="contact-stats">
                <span class="contact-item">
                    <i class="material-icons contact-icon">phone</i>
                    ${cobrador.telefono}
                </span>
                <span class="contact-item">
                    <i class="material-icons contact-icon">email</i>
                    ${cobrador.email}
                </span>
            </div>
            <div class="chart-container" style="height: 250px; padding: 10px;">
                <canvas id="${canvasId}"></canvas>
            </div>
        `;

        // Inicializar el ripple en el botón de opciones
        const iconButton = card.querySelector('.mdc-icon-button');
        if (iconButton) {
            new mdc.ripple.MDCRipple(iconButton).unbounded = true;
        }

        return { card, canvasId };
    }

    // Filtra los cobradores según los criterios de búsqueda
    filterCobradores() {
        const searchText = this.searchField ? this.searchField.value.toLowerCase() : '';
        const selectedZone = this.zoneSelect ? this.zoneSelect.value : 'todas';

        const filteredCobradores = cobradores.filter(cobrador => {
            const matchesSearch = !searchText || 
                cobrador.nombre.toLowerCase().includes(searchText) ||
                cobrador.email.toLowerCase().includes(searchText);
            
            const matchesZone = selectedZone === 'todas' || cobrador.zona === selectedZone;
            
            return matchesSearch && matchesZone;
        });

        this.renderCobradores(filteredCobradores);
    }

    // Renderiza las tarjetas de los cobradores
    renderCobradores(cobradoresToRender = cobradores) {
        if (!this.gridContainer) return;

        // Limpiar los gráficos existentes
        this.charts.forEach(chart => chart.destroy());
        this.charts.clear();

        // Limpiar el contenedor
        this.gridContainer.innerHTML = '';

        // Crear y agregar las tarjetas
        cobradoresToRender.forEach((cobrador, index) => {
            const { card, canvasId } = this.createCobradorCard(cobrador, index);
            this.gridContainer.appendChild(card);

            // Crear el gráfico después de que la tarjeta esté en el DOM
            if (cobrador.estadisticas) {
                requestAnimationFrame(() => {
                    this.createChart(canvasId, cobrador.estadisticas);
                });
            }
        });
    }
}

// Inicialización cuando el DOM esté listo
document.addEventListener('DOMContentLoaded', () => {
    try {
        const ui = new CobradoresUI();
        ui.initializeMDC();
        ui.renderCobradores();
    } catch (error) {
        console.error('Error durante la inicialización:', error);
    }
});

// Función para generar las tarjetas de cobradores
function generarTarjetasCobrador() {
    const cobradoresGrid = document.querySelector('.cobradores-grid');
    
    cobradoresGrid.innerHTML = cobradores.map((cobrador, index) => `
        <div class="cobrador-card mdc-card">
            <div class="cobrador-header">
                <div class="cobrador-avatar">
                    ${cobrador.iniciales}
                </div>
                <div class="cobrador-info">
                    <h3 class="cobrador-name">${cobrador.nombre}</h3>
                    <p class="cobrador-zone">${cobrador.zona}</p>
                </div>
                <button class="mdc-icon-button material-icons">more_vert</button>
            </div>
            
            <div class="cobrador-stats">
                <div class="total-amount">
                    <div class="stat-value">${cobrador.total}</div>
                    <div class="stat-label">Total</div>
                </div>
                
                <div class="stats-row">
                    <div class="stat-item">
                        <div class="stat-value">${cobrador.cobranzas}</div>
                        <div class="stat-label">Cobranzas</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-value">${cobrador.efectividad}</div>
                        <div class="stat-label">Efectividad</div>
                    </div>
                </div>
            </div>

            <div class="stats-chart">
                <canvas id="chart-${index}" height="100"></canvas>
            </div>

            <div class="contact-stats">
                <a href="tel:${cobrador.telefono}" class="contact-item">
                    <i class="material-icons contact-icon">phone</i>
                    ${cobrador.telefono}
                </a>
                <a href="mailto:${cobrador.email}" class="contact-item">
                    <i class="material-icons contact-icon">email</i>
                    ${cobrador.email}
                </a>
            </div>
        </div>
    `).join('');

    // Inicializar los gráficos
    cobradores.forEach((cobrador, index) => {
        const ctx = document.getElementById(`chart-${index}`).getContext('2d');
        new Chart(ctx, {
            type: 'line',
            data: {
                labels: cobrador.estadisticas_semana.map(stat => {
                    const fecha = new Date(stat.fecha);
                    return fecha.toLocaleDateString('es-ES', { weekday: 'short' });
                }),
                datasets: [
                    {
                        label: 'Cobranzas',
                        data: cobrador.estadisticas_semana.map(stat => stat.cobranzas),
                        borderColor: 'rgb(75, 192, 192)',
                        tension: 0.1,
                        fill: false
                    },
                    {
                        label: 'Efectividad %',
                        data: cobrador.estadisticas_semana.map(stat => stat.efectividad),
                        borderColor: 'rgb(255, 99, 132)',
                        tension: 0.1,
                        fill: false
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                        labels: {
                            boxWidth: 20,
                            padding: 10,
                            font: {
                                size: 11
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            font: {
                                size: 10
                            }
                        }
                    },
                    x: {
                        ticks: {
                            font: {
                                size: 10
                            }
                        }
                    }
                }
            }
        });
    });

    // Inicializar los botones de Material Design
    document.querySelectorAll('.mdc-icon-button').forEach(button => {
        new mdc.ripple.MDCRipple(button).unbounded = true;
    });
}

// Ejecutar cuando el DOM esté listo
document.addEventListener('DOMContentLoaded', function() {
    generarTarjetasCobrador();
}); 